// Stridevest Limited - Main JavaScript

// Auto-hide alerts after 5 seconds
document.addEventListener('DOMContentLoaded', function() {
    const alerts = document.querySelectorAll('.alert');
    alerts.forEach(alert => {
        setTimeout(() => {
            alert.style.animation = 'slideOut 0.3s ease';
            setTimeout(() => alert.remove(), 300);
        }, 5000);
    });
});

// Form validation
function validateForm(formId) {
    const form = document.getElementById(formId);
    if (!form) return true;
    
    const requiredFields = form.querySelectorAll('[required]');
    let isValid = true;
    
    requiredFields.forEach(field => {
        if (!field.value.trim()) {
            field.style.borderColor = 'var(--error)';
            isValid = false;
        } else {
            field.style.borderColor = 'var(--color-border)';
        }
    });
    
    return isValid;
}

// Paystack Payment Integration
function initializePayment(email, amount, reference, callback) {
    const handler = PaystackPop.setup({
        key: window.paystackPublicKey,
        email: email,
        amount: amount,
        ref: reference,
        callback: function(response) {
            if (callback) callback(response);
        },
        onClose: function() {
            console.log('Payment window closed');
        }
    });
    handler.openIframe();
}

// File upload preview
function previewFile(input, previewId) {
    const file = input.files[0];
    const preview = document.getElementById(previewId);
    
    if (file && preview) {
        const reader = new FileReader();
        reader.onload = function(e) {
            if (file.type.startsWith('image/')) {
                preview.innerHTML = `<img src="${e.target.result}" style="max-width: 200px; border-radius: 8px;">`;
            } else {
                preview.innerHTML = `<p>File selected: ${file.name}</p>`;
            }
        };
        reader.readAsDataURL(file);
    }
}

// Confirm action
function confirmAction(message) {
    return confirm(message || 'Are you sure you want to proceed?');
}

// Format currency
function formatCurrency(amount) {
    return new Intl.NumberFormat('en-NG', {
        style: 'currency',
        currency: 'NGN'
    }).format(amount);
}

// AJAX helper
async function ajaxRequest(url, method = 'GET', data = null) {
    const options = {
        method: method,
        headers: {
            'Content-Type': 'application/json',
        }
    };
    
    if (data) {
        options.body = JSON.stringify(data);
    }
    
    try {
        const response = await fetch(url, options);
        return await response.json();
    } catch (error) {
        console.error('AJAX Error:', error);
        return null;
    }
}

// Mark notification as read
async function markNotificationRead(notificationId) {
    await ajaxRequest(`/api/notifications/${notificationId}/read`, 'POST');
}

// Real-time notification polling (every 30 seconds)
if (document.querySelector('.notifications-list')) {
    setInterval(async () => {
        const notifications = await ajaxRequest('/api/notifications/unread');
        if (notifications && notifications.length > 0) {
            updateNotificationBadge(notifications.length);
        }
    }, 30000);
}

function updateNotificationBadge(count) {
    const badge = document.querySelector('.notification-badge');
    if (badge) {
        badge.textContent = count;
        badge.style.display = count > 0 ? 'block' : 'none';
    }
}

// Smooth scroll
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function (e) {
        e.preventDefault();
        const target = document.querySelector(this.getAttribute('href'));
        if (target) {
            target.scrollIntoView({ behavior: 'smooth' });
        }
    });
});

// Mobile menu toggle
const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
const navbarMenu = document.querySelector('.navbar-menu');

if (mobileMenuBtn && navbarMenu) {
    mobileMenuBtn.addEventListener('click', () => {
        navbarMenu.classList.toggle('active');
    });
}

// Dynamic form field management (Admin)
function addFormField() {
    const container = document.getElementById('form-fields-container');
    const fieldCount = container.children.length;
    
    const fieldHTML = `
        <div class="form-field-item" data-field="${fieldCount}">
            <input type="text" name="field_name[]" placeholder="Field Name" required>
            <input type="text" name="field_label[]" placeholder="Field Label" required>
            <select name="field_type[]" required>
                <option value="text">Text</option>
                <option value="textarea">Textarea</option>
                <option value="select">Select</option>
                <option value="checkbox">Checkbox</option>
                <option value="file">File</option>
            </select>
            <button type="button" onclick="removeFormField(${fieldCount})">Remove</button>
        </div>
    `;
    
    container.insertAdjacentHTML('beforeend', fieldHTML);
}

function removeFormField(fieldId) {
    const field = document.querySelector(`[data-field="${fieldId}"]`);
    if (field) field.remove();
}

// Chart.js integration for admin reports (if Chart.js is loaded)
function renderRevenueChart(data) {
    const ctx = document.getElementById('revenueChart');
    if (!ctx || typeof Chart === 'undefined') return;
    
    new Chart(ctx, {
        type: 'line',
        data: {
            labels: data.labels,
            datasets: [{
                label: 'Revenue',
                data: data.values,
                borderColor: 'rgb(0, 102, 255)',
                backgroundColor: 'rgba(0, 102, 255, 0.1)',
                tension: 0.4
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: {
                    display: false
                }
            }
        }
    });
}

// Export to CSV
function exportTableToCSV(tableId, filename) {
    const table = document.getElementById(tableId);
    if (!table) return;
    
    let csv = [];
    const rows = table.querySelectorAll('tr');
    
    rows.forEach(row => {
        const cols = row.querySelectorAll('td, th');
        const rowData = Array.from(cols).map(col => col.textContent);
        csv.push(rowData.join(','));
    });
    
    const csvContent = csv.join('\n');
    const blob = new Blob([csvContent], { type: 'text/csv' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename || 'export.csv';
    a.click();
}

console.log('Stridevest Limited - Application Loaded');
