<?php

use App\Core\Session;
use App\Core\Auth;

function redirect($url) {
    header("Location: $url");
    exit;
}

function back() {
    redirect($_SERVER['HTTP_REFERER'] ?? '/');
}

function view($viewName, $data = []) {
    extract($data);
    $viewPath = __DIR__ . '/../Views/' . str_replace('.', '/', $viewName) . '.php';
    
    if (file_exists($viewPath)) {
        require $viewPath;
    } else {
        die("View not found: $viewName");
    }
}

function asset($path) {
    return '/public/' . ltrim($path, '/');
}

function url($path = '') {
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    return $protocol . '://' . $host . '/' . ltrim($path, '/');
}

function old($key, $default = '') {
    return Session::get('old_' . $key, $default);
}

function csrf_field() {
    $token = Session::csrfToken();
    return '<input type="hidden" name="csrf_token" value="' . $token . '">';
}

function csrf_token() {
    return Session::csrfToken();
}

function auth() {
    return Auth::user();
}

function formatMoney($amount, $currency = 'NGN') {
    return $currency . ' ' . number_format($amount, 2);
}

function formatDate($date, $format = 'M d, Y') {
    return date($format, strtotime($date));
}

function sanitize($data) {
    if (is_array($data)) {
        return array_map('sanitize', $data);
    }
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}

function validate($data, $rules) {
    $errors = [];
    
    foreach ($rules as $field => $ruleSet) {
        $ruleArray = explode('|', $ruleSet);
        $value = $data[$field] ?? null;
        
        foreach ($ruleArray as $rule) {
            if ($rule === 'required' && empty($value)) {
                $errors[$field] = ucfirst($field) . ' is required';
                break;
            }
            
            if (strpos($rule, 'min:') === 0 && strlen($value) < (int)substr($rule, 4)) {
                $errors[$field] = ucfirst($field) . ' must be at least ' . substr($rule, 4) . ' characters';
                break;
            }
            
            if (strpos($rule, 'max:') === 0 && strlen($value) > (int)substr($rule, 4)) {
                $errors[$field] = ucfirst($field) . ' must not exceed ' . substr($rule, 4) . ' characters';
                break;
            }
            
            if ($rule === 'email' && !filter_var($value, FILTER_VALIDATE_EMAIL)) {
                $errors[$field] = ucfirst($field) . ' must be a valid email address';
                break;
            }
            
            if ($rule === 'numeric' && !is_numeric($value)) {
                $errors[$field] = ucfirst($field) . ' must be a number';
                break;
            }
        }
    }
    
    return $errors;
}

function uploadFile($file, $directory = 'uploads') {
    if (!isset($file['error']) || $file['error'] !== UPLOAD_ERR_OK) {
        return false;
    }
    
    $uploadDir = __DIR__ . '/../../public/' . $directory . '/';
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = uniqid() . '_' . time() . '.' . $extension;
    $destination = $uploadDir . $filename;
    
    if (move_uploaded_file($file['tmp_name'], $destination)) {
        return $directory . '/' . $filename;
    }
    
    return false;
}

function sendEmail($to, $subject, $body, $from = null) {
    $config = require __DIR__ . '/../../config/app.php';
    $from = $from ?? $config['email'];
    
    $headers = "From: {$config['name']} <$from>\r\n";
    $headers .= "Reply-To: $from\r\n";
    $headers .= "MIME-Version: 1.0\r\n";
    $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
    
    return mail($to, $subject, $body, $headers);
}

function generateApplicationNumber() {
    return 'APP-' . date('Y') . '-' . strtoupper(substr(uniqid(), -8));
}

function generateReferralCode($userId) {
    return 'REF-' . strtoupper(substr(md5($userId . time()), 0, 8));
}

function logActivity($adminId, $action, $entityType = null, $entityId = null, $oldValues = null, $newValues = null) {
    $db = \App\Core\Database::getInstance();
    
    $db->execute(
        "INSERT INTO admin_logs (admin_id, action, entity_type, entity_id, old_values, new_values, ip_address, user_agent) 
         VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
        [
            $adminId,
            $action,
            $entityType,
            $entityId,
            $oldValues ? json_encode($oldValues) : null,
            $newValues ? json_encode($newValues) : null,
            $_SERVER['REMOTE_ADDR'] ?? null,
            $_SERVER['HTTP_USER_AGENT'] ?? null
        ]
    );
}

function createNotification($userId, $title, $message, $type = 'info', $actionUrl = null) {
    $db = \App\Core\Database::getInstance();
    
    $db->execute(
        "INSERT INTO notifications (user_id, title, message, type, action_url) VALUES (?, ?, ?, ?, ?)",
        [$userId, $title, $message, $type, $actionUrl]
    );
}

function calculateROI($investmentAmount, $returnAmount) {
    if ($investmentAmount == 0) return 0;
    return (($returnAmount - $investmentAmount) / $investmentAmount) * 100;
}

function distributeProfits($investmentId, $totalProfit) {
    $investorShare = $totalProfit * 0.40;
    $platformShare = $totalProfit * 0.20;
    $entrepreneurShare = $totalProfit * 0.40;
    
    $db = \App\Core\Database::getInstance();
    $db->execute(
        "INSERT INTO profit_sharing (investment_id, total_profit, investor_share, platform_share, entrepreneur_share) 
         VALUES (?, ?, ?, ?, ?)",
        [$investmentId, $totalProfit, $investorShare, $platformShare, $entrepreneurShare]
    );
    
    return [
        'investor' => $investorShare,
        'platform' => $platformShare,
        'entrepreneur' => $entrepreneurShare
    ];
}

function timeAgo($datetime) {
    $timestamp = strtotime($datetime);
    $difference = time() - $timestamp;
    
    if ($difference < 60) return 'just now';
    if ($difference < 3600) return floor($difference / 60) . ' minutes ago';
    if ($difference < 86400) return floor($difference / 3600) . ' hours ago';
    if ($difference < 604800) return floor($difference / 86400) . ' days ago';
    
    return date('M d, Y', $timestamp);
}
