<?php
namespace App\Controllers;

use App\Core\Database;
use App\Core\Auth;
use App\Core\Session;

class MentorController {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    public function dashboard() {
        $userId = Auth::id();
        
        // Get mentorships
        $mentorships = $this->db->fetchAll(
            "SELECT m.*, u.first_name, u.last_name, u.email, u.profile_image, a.business_name 
             FROM mentorships m 
             JOIN users u ON m.mentee_id = u.id 
             LEFT JOIN applications a ON m.application_id = a.id 
             WHERE m.mentor_id = ? 
             ORDER BY m.created_at DESC",
            [$userId]
        );
        
        // Get upcoming sessions
        $upcomingSessions = $this->db->fetchAll(
            "SELECT ms.*, m.mentee_id, u.first_name, u.last_name 
             FROM mentorship_sessions ms 
             JOIN mentorships m ON ms.mentorship_id = m.id 
             JOIN users u ON m.mentee_id = u.id 
             WHERE m.mentor_id = ? AND ms.status = 'scheduled' AND ms.session_date > NOW() 
             ORDER BY ms.session_date ASC LIMIT 5",
            [$userId]
        );
        
        // Get statistics
        $stats = [
            'total_mentees' => count($mentorships),
            'active_mentorships' => $this->db->fetchOne(
                "SELECT COUNT(*) as count FROM mentorships WHERE mentor_id = ? AND status = 'active'",
                [$userId]
            )['count'],
            'total_sessions' => $this->db->fetchOne(
                "SELECT COUNT(*) as count FROM mentorship_sessions ms 
                 JOIN mentorships m ON ms.mentorship_id = m.id 
                 WHERE m.mentor_id = ?",
                [$userId]
            )['count'],
            'avg_rating' => $this->db->fetchOne(
                "SELECT AVG(mentor_rating) as avg FROM mentorships WHERE mentor_id = ? AND mentor_rating IS NOT NULL",
                [$userId]
            )['avg'] ?? 0
        ];
        
        view('mentor.dashboard', [
            'mentorships' => $mentorships,
            'upcomingSessions' => $upcomingSessions,
            'stats' => $stats
        ]);
    }
    
    public function mentees() {
        $userId = Auth::id();
        
        $mentorships = $this->db->fetchAll(
            "SELECT m.*, u.first_name, u.last_name, u.email, u.phone, u.profile_image, 
             a.business_name, a.industry 
             FROM mentorships m 
             JOIN users u ON m.mentee_id = u.id 
             LEFT JOIN applications a ON m.application_id = a.id 
             WHERE m.mentor_id = ? 
             ORDER BY m.created_at DESC",
            [$userId]
        );
        
        view('mentor.mentees', [
            'mentorships' => $mentorships
        ]);
    }
    
    public function viewMentee($mentorshipId) {
        $userId = Auth::id();
        
        $mentorship = $this->db->fetchOne(
            "SELECT m.*, u.first_name, u.last_name, u.email, u.phone, u.profile_image, 
             a.business_name, a.industry, a.form_data 
             FROM mentorships m 
             JOIN users u ON m.mentee_id = u.id 
             LEFT JOIN applications a ON m.application_id = a.id 
             WHERE m.id = ? AND m.mentor_id = ?",
            [$mentorshipId, $userId]
        );
        
        if (!$mentorship) {
            Session::flash('error', 'Mentorship not found');
            redirect('/mentor/mentees');
        }
        
        // Get sessions
        $sessions = $this->db->fetchAll(
            "SELECT * FROM mentorship_sessions WHERE mentorship_id = ? ORDER BY session_date DESC",
            [$mentorshipId]
        );
        
        view('mentor.view-mentee', [
            'mentorship' => $mentorship,
            'sessions' => $sessions
        ]);
    }
    
    public function scheduleSession() {
        if (!Session::verifyCsrf($_POST['csrf_token'] ?? '')) {
            Session::flash('error', 'Invalid request');
            back();
        }
        
        $mentorshipId = sanitize($_POST['mentorship_id']);
        $sessionTitle = sanitize($_POST['session_title']);
        $sessionDate = sanitize($_POST['session_date']);
        $duration = sanitize($_POST['duration_minutes'] ?? 60);
        $sessionType = sanitize($_POST['session_type']);
        $meetingLink = sanitize($_POST['meeting_link'] ?? '');
        
        // Verify mentorship belongs to this mentor
        $mentorship = $this->db->fetchOne(
            "SELECT * FROM mentorships WHERE id = ? AND mentor_id = ?",
            [$mentorshipId, Auth::id()]
        );
        
        if (!$mentorship) {
            Session::flash('error', 'Invalid mentorship');
            back();
        }
        
        $this->db->execute(
            "INSERT INTO mentorship_sessions (mentorship_id, session_title, session_date, duration_minutes, session_type, meeting_link) 
             VALUES (?, ?, ?, ?, ?, ?)",
            [$mentorshipId, $sessionTitle, $sessionDate, $duration, $sessionType, $meetingLink]
        );
        
        // Notify mentee
        createNotification(
            $mentorship['mentee_id'],
            'New Mentorship Session Scheduled',
            "Your mentor has scheduled a session: $sessionTitle on " . formatDate($sessionDate, 'M d, Y h:i A'),
            'info',
            "/entrepreneur/mentorship/$mentorshipId"
        );
        
        Session::flash('success', 'Session scheduled successfully');
        back();
    }
    
    public function completeSession() {
        if (!Session::verifyCsrf($_POST['csrf_token'] ?? '')) {
            Session::flash('error', 'Invalid request');
            back();
        }
        
        $sessionId = sanitize($_POST['session_id']);
        $notes = sanitize($_POST['notes'] ?? '');
        
        $this->db->execute(
            "UPDATE mentorship_sessions SET status = 'completed', notes = ? WHERE id = ?",
            [$notes, $sessionId]
        );
        
        // Update mentorship sessions count
        $session = $this->db->fetchOne("SELECT mentorship_id FROM mentorship_sessions WHERE id = ?", [$sessionId]);
        $this->db->execute(
            "UPDATE mentorships SET sessions_completed = sessions_completed + 1 WHERE id = ?",
            [$session['mentorship_id']]
        );
        
        Session::flash('success', 'Session marked as completed');
        back();
    }
    
    public function resources() {
        $userId = Auth::id();
        
        $resources = $this->db->fetchAll(
            "SELECT * FROM mentor_resources WHERE mentor_id = ? ORDER BY created_at DESC",
            [$userId]
        );
        
        view('mentor.resources', [
            'resources' => $resources
        ]);
    }
    
    public function uploadResource() {
        if (!Session::verifyCsrf($_POST['csrf_token'] ?? '')) {
            Session::flash('error', 'Invalid request');
            back();
        }
        
        $title = sanitize($_POST['title']);
        $description = sanitize($_POST['description'] ?? '');
        $resourceType = sanitize($_POST['resource_type']);
        $category = sanitize($_POST['category'] ?? '');
        $url = sanitize($_POST['url'] ?? '');
        $isPublic = isset($_POST['is_public']) ? true : false;
        
        $filePath = null;
        if (isset($_FILES['file']) && $_FILES['file']['error'] === UPLOAD_ERR_OK) {
            $filePath = uploadFile($_FILES['file'], 'uploads/resources');
        }
        
        $this->db->execute(
            "INSERT INTO mentor_resources (mentor_id, title, description, resource_type, file_path, url, category, is_public) 
             VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
            [Auth::id(), $title, $description, $resourceType, $filePath, $url, $category, $isPublic]
        );
        
        Session::flash('success', 'Resource uploaded successfully');
        redirect('/mentor/resources');
    }
}
