<?php
namespace App\Controllers;

use App\Core\Database;
use App\Core\Auth;
use App\Core\Session;

class InvestorController {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    public function dashboard() {
        $userId = Auth::id();
        
        // Get wallet
        $wallet = $this->db->fetchOne(
            "SELECT * FROM wallets WHERE user_id = ?",
            [$userId]
        );
        
        // Get investments
        $investments = $this->db->fetchAll(
            "SELECT i.*, a.business_name, a.application_number, a.industry 
             FROM investments i 
             JOIN applications a ON i.application_id = a.id 
             WHERE i.investor_id = ? 
             ORDER BY i.created_at DESC",
            [$userId]
        );
        
        // Calculate total ROI
        $totalInvested = 0;
        $totalReturns = 0;
        foreach ($investments as $investment) {
            $totalInvested += $investment['amount'];
            $totalReturns += $investment['actual_return_amount'] ?? 0;
        }
        
        // Get notifications
        $notifications = $this->db->fetchAll(
            "SELECT * FROM notifications WHERE user_id = ? AND is_read = FALSE ORDER BY created_at DESC LIMIT 5",
            [$userId]
        );
        
        view('investor.dashboard', [
            'wallet' => $wallet,
            'investments' => $investments,
            'totalInvested' => $totalInvested,
            'totalReturns' => $totalReturns,
            'notifications' => $notifications
        ]);
    }
    
    public function browseStartups() {
        // Get approved and funded applications
        $startups = $this->db->fetchAll(
            "SELECT a.*, u.first_name, u.last_name, 
             (SELECT COUNT(*) FROM investments WHERE application_id = a.id) as investor_count,
             (SELECT SUM(amount) FROM investments WHERE application_id = a.id) as total_invested
             FROM applications a 
             JOIN users u ON a.user_id = u.id 
             WHERE a.status IN ('approved', 'funded') AND a.payment_status = 'paid'
             ORDER BY a.created_at DESC"
        );
        
        view('investor.browse-startups', [
            'startups' => $startups
        ]);
    }
    
    public function viewStartup($applicationId) {
        $startup = $this->db->fetchOne(
            "SELECT a.*, u.first_name, u.last_name, u.email, u.phone 
             FROM applications a 
             JOIN users u ON a.user_id = u.id 
             WHERE a.id = ? AND a.status IN ('approved', 'funded')",
            [$applicationId]
        );
        
        if (!$startup) {
            Session::flash('error', 'Startup not found');
            redirect('/investor/browse-startups');
        }
        
        // Get existing investments
        $investments = $this->db->fetchAll(
            "SELECT i.*, u.first_name, u.last_name 
             FROM investments i 
             JOIN users u ON i.investor_id = u.id 
             WHERE i.application_id = ?",
            [$applicationId]
        );
        
        // Get investment pool if exists
        $pool = $this->db->fetchOne(
            "SELECT * FROM investment_pools WHERE application_id = ?",
            [$applicationId]
        );
        
        view('investor.view-startup', [
            'startup' => $startup,
            'investments' => $investments,
            'pool' => $pool
        ]);
    }
    
    public function invest() {
        if (!Session::verifyCsrf($_POST['csrf_token'] ?? '')) {
            Session::flash('error', 'Invalid request');
            back();
        }
        
        $userId = Auth::id();
        $applicationId = sanitize($_POST['application_id']);
        $amount = sanitize($_POST['amount']);
        
        // Validate
        if (!is_numeric($amount) || $amount <= 0) {
            Session::flash('error', 'Invalid investment amount');
            back();
        }
        
        // Check wallet balance
        $wallet = $this->db->fetchOne(
            "SELECT * FROM wallets WHERE user_id = ?",
            [$userId]
        );
        
        if ($wallet['balance'] < $amount) {
            Session::flash('error', 'Insufficient wallet balance');
            back();
        }
        
        // Check if startup exists and is approved
        $application = $this->db->fetchOne(
            "SELECT * FROM applications WHERE id = ? AND status IN ('approved', 'funded')",
            [$applicationId]
        );
        
        if (!$application) {
            Session::flash('error', 'Invalid startup');
            back();
        }
        
        $this->db->beginTransaction();
        
        try {
            // Create investment
            $this->db->execute(
                "INSERT INTO investments (investor_id, application_id, amount, expected_roi_percentage) 
                 VALUES (?, ?, ?, ?)",
                [$userId, $applicationId, $amount, 15] // Default 15% expected ROI
            );
            
            $investmentId = $this->db->lastInsertId();
            
            // Update wallet
            $this->db->execute(
                "UPDATE wallets SET balance = balance - ?, total_investments = total_investments + ? 
                 WHERE user_id = ?",
                [$amount, $amount, $userId]
            );
            
            // Create transaction
            $this->db->execute(
                "INSERT INTO transactions (user_id, transaction_type, amount, status, description, related_id) 
                 VALUES (?, 'investment', ?, 'completed', ?, ?)",
                [$userId, $amount, "Investment in {$application['business_name']}", $investmentId]
            );
            
            // Update investment pool if exists
            $this->db->execute(
                "UPDATE investment_pools SET raised_amount = raised_amount + ? 
                 WHERE application_id = ?",
                [$amount, $applicationId]
            );
            
            // Notify entrepreneur
            createNotification(
                $application['user_id'],
                'New Investment Received',
                "You have received an investment of " . formatMoney($amount) . " for {$application['business_name']}",
                'success',
                "/entrepreneur/view-application/$applicationId"
            );
            
            $this->db->commit();
            
            Session::flash('success', 'Investment successful!');
            redirect('/investor/dashboard');
            
        } catch (\Exception $e) {
            $this->db->rollback();
            Session::flash('error', 'Investment failed. Please try again.');
            back();
        }
    }
    
    public function wallet() {
        $userId = Auth::id();
        
        $wallet = $this->db->fetchOne(
            "SELECT * FROM wallets WHERE user_id = ?",
            [$userId]
        );
        
        $transactions = $this->db->fetchAll(
            "SELECT * FROM transactions WHERE user_id = ? ORDER BY created_at DESC LIMIT 50",
            [$userId]
        );
        
        view('investor.wallet', [
            'wallet' => $wallet,
            'transactions' => $transactions
        ]);
    }
    
    public function depositFunds() {
        if (!Session::verifyCsrf($_POST['csrf_token'] ?? '')) {
            Session::flash('error', 'Invalid request');
            back();
        }
        
        $userId = Auth::id();
        $amount = sanitize($_POST['amount']);
        
        if (!is_numeric($amount) || $amount < 1000) {
            Session::flash('error', 'Minimum deposit is ₦1,000');
            back();
        }
        
        // Generate payment reference
        $reference = 'DEP-' . time() . '-' . $userId;
        
        // Create pending transaction
        $this->db->execute(
            "INSERT INTO transactions (user_id, transaction_type, amount, status, payment_reference, description) 
             VALUES (?, 'deposit', ?, 'pending', ?, 'Wallet deposit')",
            [$userId, $amount, $reference]
        );
        
        // Return payment data for Paystack
        echo json_encode([
            'success' => true,
            'reference' => $reference,
            'amount' => $amount * 100 // Paystack uses kobo
        ]);
    }
    
    public function verifyDeposit() {
        $reference = $_GET['reference'] ?? '';
        $config = require __DIR__ . '/../../config/app.php';
        
        // Verify with Paystack
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, "https://api.paystack.co/transaction/verify/" . $reference);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "Authorization: Bearer " . $config['paystack_secret_key']
        ]);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        $result = json_decode($response, true);
        
        if ($result['status'] && $result['data']['status'] === 'success') {
            $userId = Auth::id();
            $amount = $result['data']['amount'] / 100; // Convert from kobo
            
            $this->db->beginTransaction();
            
            try {
                // Update transaction
                $this->db->execute(
                    "UPDATE transactions SET status = 'completed', gateway_response = ? 
                     WHERE payment_reference = ? AND user_id = ?",
                    [json_encode($result), $reference, $userId]
                );
                
                // Update wallet
                $this->db->execute(
                    "UPDATE wallets SET balance = balance + ?, total_deposits = total_deposits + ? 
                     WHERE user_id = ?",
                    [$amount, $amount, $userId]
                );
                
                $this->db->commit();
                
                Session::flash('success', 'Deposit successful! Your wallet has been credited.');
                redirect('/investor/wallet');
                
            } catch (\Exception $e) {
                $this->db->rollback();
                Session::flash('error', 'Deposit processing failed');
                redirect('/investor/wallet');
            }
        } else {
            Session::flash('error', 'Payment verification failed');
            redirect('/investor/wallet');
        }
    }
}
