<?php
namespace App\Controllers;

use App\Core\Database;
use App\Core\Auth;
use App\Core\Session;

class EntrepreneurController {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    public function dashboard() {
        $userId = Auth::id();
        
        // Get applications
        $applications = $this->db->fetchAll(
            "SELECT * FROM applications WHERE user_id = ? ORDER BY created_at DESC",
            [$userId]
        );
        
        // Get mentorships
        $mentorships = $this->db->fetchAll(
            "SELECT m.*, u.first_name, u.last_name, u.profile_image 
             FROM mentorships m 
             JOIN users u ON m.mentor_id = u.id 
             WHERE m.mentee_id = ? AND m.status = 'active'",
            [$userId]
        );
        
        // Get notifications
        $notifications = $this->db->fetchAll(
            "SELECT * FROM notifications WHERE user_id = ? AND is_read = FALSE ORDER BY created_at DESC LIMIT 5",
            [$userId]
        );
        
        // Get wallet
        $wallet = $this->db->fetchOne(
            "SELECT * FROM wallets WHERE user_id = ?",
            [$userId]
        );
        
        view('entrepreneur.dashboard', [
            'applications' => $applications,
            'mentorships' => $mentorships,
            'notifications' => $notifications,
            'wallet' => $wallet
        ]);
    }
    
    public function showApplicationForm() {
        // Get dynamic form fields
        $formFields = $this->db->fetchAll(
            "SELECT * FROM form_fields WHERE is_active = TRUE ORDER BY field_order ASC"
        );
        
        view('entrepreneur.application-form', [
            'formFields' => $formFields
        ]);
    }
    
    public function submitApplication() {
        if (!Session::verifyCsrf($_POST['csrf_token'] ?? '')) {
            Session::flash('error', 'Invalid request');
            back();
        }
        
        $userId = Auth::id();
        $config = require __DIR__ . '/../../config/app.php';
        
        // Validate basic fields
        $errors = validate($_POST, [
            'business_name' => 'required|min:3|max:255',
            'business_type' => 'required',
            'industry' => 'required',
            'funding_amount' => 'required|numeric'
        ]);
        
        if (!empty($errors)) {
            Session::flash('errors', $errors);
            back();
        }
        
        // Handle file uploads
        $pitchDeck = null;
        $businessPlan = null;
        $cacDocument = null;
        
        if (isset($_FILES['pitch_deck']) && $_FILES['pitch_deck']['error'] === UPLOAD_ERR_OK) {
            $pitchDeck = uploadFile($_FILES['pitch_deck'], 'uploads/pitch_decks');
        }
        
        if (isset($_FILES['business_plan']) && $_FILES['business_plan']['error'] === UPLOAD_ERR_OK) {
            $businessPlan = uploadFile($_FILES['business_plan'], 'uploads/documents');
        }
        
        if (isset($_FILES['cac_document']) && $_FILES['cac_document']['error'] === UPLOAD_ERR_OK) {
            $cacDocument = uploadFile($_FILES['cac_document'], 'uploads/documents');
        }
        
        // Collect dynamic form data
        $formData = [];
        $formFields = $this->db->fetchAll(
            "SELECT * FROM form_fields WHERE is_active = TRUE"
        );
        
        foreach ($formFields as $field) {
            $fieldName = $field['field_name'];
            if (isset($_POST[$fieldName])) {
                $formData[$fieldName] = sanitize($_POST[$fieldName]);
            }
        }
        
        // Generate application number
        $applicationNumber = generateApplicationNumber();
        
        // Create application
        $this->db->execute(
            "INSERT INTO applications (user_id, application_number, business_name, business_type, industry, 
             form_data, pitch_deck, business_plan, cac_document, funding_amount, is_draft) 
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
            [
                $userId,
                $applicationNumber,
                sanitize($_POST['business_name']),
                sanitize($_POST['business_type']),
                sanitize($_POST['industry']),
                json_encode($formData),
                $pitchDeck,
                $businessPlan,
                $cacDocument,
                sanitize($_POST['funding_amount']),
                false
            ]
        );
        
        $applicationId = $this->db->lastInsertId();
        
        // Create notification for payment
        createNotification(
            $userId,
            'Application Submitted',
            "Your application ($applicationNumber) has been created. Please complete the payment of " . formatMoney($config['application_fee']) . " to submit.",
            'info',
            "/entrepreneur/pay-application/$applicationId"
        );
        
        Session::flash('success', 'Application created! Please complete payment to submit.');
        redirect("/entrepreneur/pay-application/$applicationId");
    }
    
    public function showPaymentPage($applicationId) {
        $userId = Auth::id();
        $config = require __DIR__ . '/../../config/app.php';
        
        $application = $this->db->fetchOne(
            "SELECT * FROM applications WHERE id = ? AND user_id = ?",
            [$applicationId, $userId]
        );
        
        if (!$application) {
            Session::flash('error', 'Application not found');
            redirect('/entrepreneur/dashboard');
        }
        
        if ($application['payment_status'] === 'paid') {
            Session::flash('info', 'Payment already completed');
            redirect('/entrepreneur/dashboard');
        }
        
        view('entrepreneur.payment', [
            'application' => $application,
            'amount' => $config['application_fee'],
            'paystackPublicKey' => $config['paystack_public_key']
        ]);
    }
    
    public function verifyPayment() {
        $reference = $_GET['reference'] ?? '';
        $config = require __DIR__ . '/../../config/app.php';
        
        // Verify with Paystack
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, "https://api.paystack.co/transaction/verify/" . $reference);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "Authorization: Bearer " . $config['paystack_secret_key']
        ]);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        $result = json_decode($response, true);
        
        if ($result['status'] && $result['data']['status'] === 'success') {
            $userId = Auth::id();
            
            // Update application
            $this->db->execute(
                "UPDATE applications SET payment_status = 'paid', payment_reference = ?, submitted_at = CURRENT_TIMESTAMP 
                 WHERE user_id = ? AND payment_reference IS NULL",
                [$reference, $userId]
            );
            
            // Create transaction record
            $this->db->execute(
                "INSERT INTO transactions (user_id, transaction_type, amount, status, payment_reference, description) 
                 VALUES (?, 'application_fee', ?, 'completed', ?, 'Application submission fee')",
                [$userId, $config['application_fee'], $reference]
            );
            
            // Notify admin
            $admins = $this->db->fetchAll("SELECT id FROM users WHERE role = 'admin'");
            foreach ($admins as $admin) {
                createNotification(
                    $admin['id'],
                    'New Application Submitted',
                    'A new application has been submitted and requires review.',
                    'info',
                    '/admin/applications'
                );
            }
            
            Session::flash('success', 'Payment successful! Your application has been submitted for review.');
            redirect('/entrepreneur/dashboard');
        } else {
            Session::flash('error', 'Payment verification failed');
            redirect('/entrepreneur/dashboard');
        }
    }
    
    public function viewApplication($applicationId) {
        $userId = Auth::id();
        
        $application = $this->db->fetchOne(
            "SELECT * FROM applications WHERE id = ? AND user_id = ?",
            [$applicationId, $userId]
        );
        
        if (!$application) {
            Session::flash('error', 'Application not found');
            redirect('/entrepreneur/dashboard');
        }
        
        view('entrepreneur.view-application', [
            'application' => $application
        ]);
    }
}
