<?php
namespace App\Controllers;

use App\Core\Database;
use App\Core\Auth;
use App\Core\Session;

class AuthController {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    public function showLogin() {
        if (Auth::check()) {
            redirect($this->getDashboardUrl());
        }
        view('auth.login');
    }
    
    public function login() {
        if (!Session::verifyCsrf($_POST['csrf_token'] ?? '')) {
            Session::flash('error', 'Invalid request');
            back();
        }
        
        $email = sanitize($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        
        $result = Auth::attempt($email, $password);
        
        if ($result === true) {
            Session::flash('success', 'Login successful');
            redirect($this->getDashboardUrl());
        } else if (is_array($result)) {
            Session::flash('error', $result['error']);
            back();
        } else {
            Session::flash('error', 'Invalid credentials');
            back();
        }
    }
    
    public function showRegister() {
        if (Auth::check()) {
            redirect($this->getDashboardUrl());
        }
        view('auth.register');
    }
    
    public function register() {
        if (!Session::verifyCsrf($_POST['csrf_token'] ?? '')) {
            Session::flash('error', 'Invalid request');
            back();
        }
        
        $errors = validate($_POST, [
            'first_name' => 'required|min:2|max:100',
            'last_name' => 'required|min:2|max:100',
            'email' => 'required|email',
            'phone' => 'required|min:10|max:20',
            'password' => 'required|min:8',
            'role' => 'required'
        ]);
        
        if (!empty($errors)) {
            Session::flash('errors', $errors);
            back();
        }
        
        // Check if email exists
        $existingUser = $this->db->fetchOne(
            "SELECT id FROM users WHERE email = ?",
            [sanitize($_POST['email'])]
        );
        
        if ($existingUser) {
            Session::flash('error', 'Email already registered');
            back();
        }
        
        // Create user
        $this->db->execute(
            "INSERT INTO users (first_name, last_name, email, phone, password, role) 
             VALUES (?, ?, ?, ?, ?, ?)",
            [
                sanitize($_POST['first_name']),
                sanitize($_POST['last_name']),
                sanitize($_POST['email']),
                sanitize($_POST['phone']),
                Auth::hashPassword($_POST['password']),
                sanitize($_POST['role'])
            ]
        );
        
        $userId = $this->db->lastInsertId();
        
        // Create wallet
        $this->db->execute(
            "INSERT INTO wallets (user_id) VALUES (?)",
            [$userId]
        );
        
        // Generate verification token
        $token = Auth::generateToken();
        $this->db->execute(
            "INSERT INTO verification_tokens (user_id, token, type, expires_at) 
             VALUES (?, ?, 'email', ?)",
            [$userId, $token, date('Y-m-d H:i:s', strtotime('+24 hours'))]
        );
        
        // Send verification email
        $verificationLink = url("verify-email?token=$token");
        sendEmail(
            $_POST['email'],
            'Verify Your Email - Stridevest',
            "Click here to verify your email: <a href='$verificationLink'>$verificationLink</a>"
        );
        
        Session::flash('success', 'Registration successful! Please check your email to verify your account.');
        redirect('/login');
    }
    
    public function verifyEmail() {
        $token = $_GET['token'] ?? '';
        
        $verification = $this->db->fetchOne(
            "SELECT * FROM verification_tokens WHERE token = ? AND type = 'email' AND used = FALSE AND expires_at > NOW()",
            [$token]
        );
        
        if (!$verification) {
            Session::flash('error', 'Invalid or expired verification link');
            redirect('/login');
        }
        
        // Update user
        $this->db->execute(
            "UPDATE users SET email_verified = TRUE WHERE id = ?",
            [$verification['user_id']]
        );
        
        // Mark token as used
        $this->db->execute(
            "UPDATE verification_tokens SET used = TRUE WHERE id = ?",
            [$verification['id']]
        );
        
        Session::flash('success', 'Email verified successfully! You can now login.');
        redirect('/login');
    }
    
    public function logout() {
        Auth::logout();
        redirect('/login');
    }
    
    private function getDashboardUrl() {
        $role = Auth::role();
        switch ($role) {
            case 'admin':
                return '/admin/dashboard';
            case 'entrepreneur':
                return '/entrepreneur/dashboard';
            case 'investor':
                return '/investor/dashboard';
            case 'mentor':
                return '/mentor/dashboard';
            default:
                return '/';
        }
    }
}
