<?php
namespace App\Controllers;

use App\Core\Database;
use App\Core\Auth;
use App\Core\Session;

class AdminController {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    public function dashboard() {
        // Get statistics
        $stats = [
            'total_users' => $this->db->fetchOne("SELECT COUNT(*) as count FROM users")['count'],
            'total_entrepreneurs' => $this->db->fetchOne("SELECT COUNT(*) as count FROM users WHERE role = 'entrepreneur'")['count'],
            'total_investors' => $this->db->fetchOne("SELECT COUNT(*) as count FROM users WHERE role = 'investor'")['count'],
            'total_mentors' => $this->db->fetchOne("SELECT COUNT(*) as count FROM users WHERE role = 'mentor'")['count'],
            'total_applications' => $this->db->fetchOne("SELECT COUNT(*) as count FROM applications")['count'],
            'pending_applications' => $this->db->fetchOne("SELECT COUNT(*) as count FROM applications WHERE status = 'pending'")['count'],
            'total_investments' => $this->db->fetchOne("SELECT COALESCE(SUM(amount), 0) as total FROM investments")['total'],
            'total_revenue' => $this->db->fetchOne("SELECT COALESCE(SUM(amount), 0) as total FROM transactions WHERE transaction_type = 'application_fee' AND status = 'completed'")['total']
        ];
        
        // Recent applications
        $recentApplications = $this->db->fetchAll(
            "SELECT a.*, u.first_name, u.last_name, u.email 
             FROM applications a 
             JOIN users u ON a.user_id = u.id 
             ORDER BY a.created_at DESC LIMIT 10"
        );
        
        // Recent users
        $recentUsers = $this->db->fetchAll(
            "SELECT * FROM users ORDER BY created_at DESC LIMIT 10"
        );
        
        view('admin.dashboard', [
            'stats' => $stats,
            'recentApplications' => $recentApplications,
            'recentUsers' => $recentUsers
        ]);
    }
    
    public function applications() {
        $status = $_GET['status'] ?? 'all';
        
        $sql = "SELECT a.*, u.first_name, u.last_name, u.email 
                FROM applications a 
                JOIN users u ON a.user_id = u.id";
        
        if ($status !== 'all') {
            $sql .= " WHERE a.status = ?";
            $applications = $this->db->fetchAll($sql . " ORDER BY a.created_at DESC", [$status]);
        } else {
            $applications = $this->db->fetchAll($sql . " ORDER BY a.created_at DESC");
        }
        
        view('admin.applications', [
            'applications' => $applications,
            'currentStatus' => $status
        ]);
    }
    
    public function viewApplication($applicationId) {
        $application = $this->db->fetchOne(
            "SELECT a.*, u.first_name, u.last_name, u.email, u.phone 
             FROM applications a 
             JOIN users u ON a.user_id = u.id 
             WHERE a.id = ?",
            [$applicationId]
        );
        
        if (!$application) {
            Session::flash('error', 'Application not found');
            redirect('/admin/applications');
        }
        
        // Get verification status
        $verification = $this->db->fetchOne(
            "SELECT * FROM business_verifications WHERE application_id = ?",
            [$applicationId]
        );
        
        view('admin.view-application', [
            'application' => $application,
            'verification' => $verification
        ]);
    }
    
    public function updateApplicationStatus() {
        if (!Session::verifyCsrf($_POST['csrf_token'] ?? '')) {
            Session::flash('error', 'Invalid request');
            back();
        }
        
        $applicationId = sanitize($_POST['application_id']);
        $status = sanitize($_POST['status']);
        $notes = sanitize($_POST['admin_notes'] ?? '');
        $rejectionReason = sanitize($_POST['rejection_reason'] ?? '');
        
        $application = $this->db->fetchOne(
            "SELECT * FROM applications WHERE id = ?",
            [$applicationId]
        );
        
        if (!$application) {
            Session::flash('error', 'Application not found');
            back();
        }
        
        // Update application
        $this->db->execute(
            "UPDATE applications SET status = ?, admin_notes = ?, rejection_reason = ?, reviewed_at = CURRENT_TIMESTAMP 
             WHERE id = ?",
            [$status, $notes, $rejectionReason, $applicationId]
        );
        
        // Log activity
        logActivity(
            Auth::id(),
            "Updated application status to $status",
            'application',
            $applicationId,
            ['status' => $application['status']],
            ['status' => $status]
        );
        
        // Notify entrepreneur
        $notificationMessage = match($status) {
            'approved' => "Your application ({$application['application_number']}) has been approved!",
            'rejected' => "Your application ({$application['application_number']}) has been rejected. Reason: $rejectionReason",
            'under_review' => "Your application ({$application['application_number']}) is now under review.",
            default => "Your application status has been updated to $status"
        };
        
        createNotification(
            $application['user_id'],
            'Application Status Updated',
            $notificationMessage,
            $status === 'approved' ? 'success' : ($status === 'rejected' ? 'error' : 'info'),
            "/entrepreneur/view-application/$applicationId"
        );
        
        Session::flash('success', 'Application status updated successfully');
        back();
    }
    
    public function formBuilder() {
        $formFields = $this->db->fetchAll(
            "SELECT * FROM form_fields ORDER BY field_order ASC"
        );
        
        view('admin.form-builder', [
            'formFields' => $formFields
        ]);
    }
    
    public function addFormField() {
        if (!Session::verifyCsrf($_POST['csrf_token'] ?? '')) {
            Session::flash('error', 'Invalid request');
            back();
        }
        
        $fieldName = sanitize($_POST['field_name']);
        $fieldLabel = sanitize($_POST['field_label']);
        $fieldType = sanitize($_POST['field_type']);
        $fieldOptions = sanitize($_POST['field_options'] ?? '');
        $isRequired = isset($_POST['is_required']) ? true : false;
        $placeholder = sanitize($_POST['placeholder'] ?? '');
        $helpText = sanitize($_POST['help_text'] ?? '');
        $fieldOrder = sanitize($_POST['field_order'] ?? 0);
        
        $this->db->execute(
            "INSERT INTO form_fields (field_name, field_label, field_type, field_options, is_required, placeholder, help_text, field_order) 
             VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
            [$fieldName, $fieldLabel, $fieldType, $fieldOptions, $isRequired, $placeholder, $helpText, $fieldOrder]
        );
        
        logActivity(Auth::id(), "Added form field: $fieldLabel", 'form_field', $this->db->lastInsertId());
        
        Session::flash('success', 'Form field added successfully');
        redirect('/admin/form-builder');
    }
    
    public function updateFormField() {
        if (!Session::verifyCsrf($_POST['csrf_token'] ?? '')) {
            Session::flash('error', 'Invalid request');
            back();
        }
        
        $fieldId = sanitize($_POST['field_id']);
        $fieldLabel = sanitize($_POST['field_label']);
        $fieldOptions = sanitize($_POST['field_options'] ?? '');
        $isRequired = isset($_POST['is_required']) ? true : false;
        $placeholder = sanitize($_POST['placeholder'] ?? '');
        $helpText = sanitize($_POST['help_text'] ?? '');
        $fieldOrder = sanitize($_POST['field_order'] ?? 0);
        $isActive = isset($_POST['is_active']) ? true : false;
        
        $this->db->execute(
            "UPDATE form_fields SET field_label = ?, field_options = ?, is_required = ?, placeholder = ?, 
             help_text = ?, field_order = ?, is_active = ? WHERE id = ?",
            [$fieldLabel, $fieldOptions, $isRequired, $placeholder, $helpText, $fieldOrder, $isActive, $fieldId]
        );
        
        logActivity(Auth::id(), "Updated form field", 'form_field', $fieldId);
        
        Session::flash('success', 'Form field updated successfully');
        redirect('/admin/form-builder');
    }
    
    public function deleteFormField($fieldId) {
        $this->db->execute("DELETE FROM form_fields WHERE id = ?", [$fieldId]);
        
        logActivity(Auth::id(), "Deleted form field", 'form_field', $fieldId);
        
        Session::flash('success', 'Form field deleted successfully');
        redirect('/admin/form-builder');
    }
    
    public function users() {
        $role = $_GET['role'] ?? 'all';
        
        $sql = "SELECT * FROM users";
        
        if ($role !== 'all') {
            $sql .= " WHERE role = ?";
            $users = $this->db->fetchAll($sql . " ORDER BY created_at DESC", [$role]);
        } else {
            $users = $this->db->fetchAll($sql . " ORDER BY created_at DESC");
        }
        
        view('admin.users', [
            'users' => $users,
            'currentRole' => $role
        ]);
    }
    
    public function updateUserStatus() {
        if (!Session::verifyCsrf($_POST['csrf_token'] ?? '')) {
            Session::flash('error', 'Invalid request');
            back();
        }
        
        $userId = sanitize($_POST['user_id']);
        $status = sanitize($_POST['status']);
        
        $this->db->execute(
            "UPDATE users SET status = ? WHERE id = ?",
            [$status, $userId]
        );
        
        logActivity(Auth::id(), "Updated user status to $status", 'user', $userId);
        
        Session::flash('success', 'User status updated successfully');
        back();
    }
    
    public function transactions() {
        $transactions = $this->db->fetchAll(
            "SELECT t.*, u.first_name, u.last_name, u.email 
             FROM transactions t 
             JOIN users u ON t.user_id = u.id 
             ORDER BY t.created_at DESC LIMIT 100"
        );
        
        view('admin.transactions', [
            'transactions' => $transactions
        ]);
    }
    
    public function reports() {
        // Monthly revenue
        $monthlyRevenue = $this->db->fetchAll(
            "SELECT DATE_TRUNC('month', created_at) as month, SUM(amount) as revenue 
             FROM transactions 
             WHERE transaction_type = 'application_fee' AND status = 'completed' 
             GROUP BY month 
             ORDER BY month DESC LIMIT 12"
        );
        
        // Investment trends
        $investmentTrends = $this->db->fetchAll(
            "SELECT DATE_TRUNC('month', investment_date) as month, SUM(amount) as total 
             FROM investments 
             GROUP BY month 
             ORDER BY month DESC LIMIT 12"
        );
        
        view('admin.reports', [
            'monthlyRevenue' => $monthlyRevenue,
            'investmentTrends' => $investmentTrends
        ]);
    }
}
